// Función helper para agregar logs visibles en la página
function addDebugLog(message, type) {
    type = type || 'info';
    var icon = '📝';
    var color = '#333';
    
    switch(type) {
        case 'success':
            icon = '✅';
            color = '#28a745';
            break;
        case 'error':
            icon = '❌';
            color = '#dc3545';
            break;
        case 'warning':
            icon = '⚠️';
            color = '#ffc107';
            break;
        case 'info':
            icon = 'ℹ️';
            color = '#17a2b8';
            break;
        case 'request':
            icon = '📤';
            color = '#007bff';
            break;
        case 'response':
            icon = '📥';
            color = '#6c757d';
            break;
    }
    
    var timestamp = new Date().toLocaleTimeString();
    var logHtml = '<div style="margin-bottom: 5px; padding: 5px; border-left: 3px solid ' + color + '; background-color: #fff;">';
    logHtml += '<span style="color: ' + color + ';">[' + timestamp + '] ' + icon + ' ' + message + '</span>';
    logHtml += '</div>';
    
    var $debugLogs = $('#debug-logs');
    if ($debugLogs.length) {
        $debugLogs.prepend(logHtml);
        // Mantener solo los últimos 50 logs
        $debugLogs.children().slice(50).remove();
    }
    
    // También loguear en consola
    console.log('[' + timestamp + '] ' + icon + ' ' + message);
}

$(function() {
    addDebugLog('Inicializando vista de documentos electrónicos...', 'info');
    console.log('🚀 Inicializando vista de documentos electrónicos...');
    console.log('URL base:', $('#url').val());
    addDebugLog('URL base: ' + $('#url').val(), 'info');
    
    // Cargar y manejar envío automático (estado como botón/badge con color)
    function actualizarLabelEnvioAutomatico(activo) {
        console.log('🏷️ [LABEL] actualizarLabelEnvioAutomatico llamado con activo:', activo);
        addDebugLog('actualizarLabelEnvioAutomatico llamado - activo: ' + activo, 'info');
        
        var $badge = $('#label_envio_automatico');
        console.log('🏷️ [LABEL] Badge encontrado:', $badge.length > 0);
        addDebugLog('Badge encontrado: ' + ($badge.length > 0 ? 'Sí' : 'No'), $badge.length > 0 ? 'success' : 'error');
        
        if ($badge.length === 0) {
            console.error('❌ [LABEL] No se encontró el elemento #label_envio_automatico');
            addDebugLog('ERROR: No se encontró el elemento #label_envio_automatico', 'error');
            return;
        }
        
        var textoAnterior = $badge.text();
        var nuevoTexto = activo ? 'Activado' : 'Desactivado';
        console.log('🏷️ [LABEL] Texto anterior:', textoAnterior, '→ Nuevo texto:', nuevoTexto);
        addDebugLog('Cambiando texto: "' + textoAnterior + '" → "' + nuevoTexto + '"', 'info');
        
        $badge.text(nuevoTexto);
        console.log('🏷️ [LABEL] Texto actualizado. Verificando:', $badge.text());
        addDebugLog('Texto actualizado. Verificado: ' + $badge.text(), 'success');
        
        $badge.removeClass('badge-secondary badge-success');
        var nuevaClase = activo ? 'badge-success' : 'badge-secondary';
        $badge.addClass(nuevaClase);
        console.log('🏷️ [LABEL] Clases actualizadas. Nueva clase:', nuevaClase);
        console.log('🏷️ [LABEL] Clases actuales del badge:', $badge.attr('class'));
        addDebugLog('Clases actualizadas. Nueva clase: ' + nuevaClase, 'success');
        
        console.log('✅ [LABEL] actualizarLabelEnvioAutomatico completado');
        addDebugLog('actualizarLabelEnvioAutomatico completado', 'success');
    }
    $.ajax({
        url: $('#url').val() + 'facturacion/documentos_config_envio',
        type: 'GET',
        dataType: 'json',
        success: function(res) {
            if (res && res.success !== false) {
                var activo = res.envio_automatico ? true : false;
                $('#switch_envio_automatico').prop('checked', activo);
                actualizarLabelEnvioAutomatico(activo);
            }
        }
    });
    // Variable para evitar múltiples diálogos simultáneos
    var procesandoCambioEnvio = false;
    
    $('#switch_envio_automatico').on('change', function(e) {
        console.log('🔄 [ENVIO_AUTO] Evento change disparado');
        addDebugLog('Evento change en switch_envio_automatico', 'info');
        
        if (procesandoCambioEnvio) {
            console.log('⚠️ [ENVIO_AUTO] Ya se está procesando un cambio, ignorando');
            addDebugLog('Cambio ya en proceso, ignorando', 'warning');
            return;
        }
        
        var $chk = $(this);
        // Leer el nuevo estado (después del cambio)
        var nuevoEstado = $chk.prop('checked');
        console.log('📊 [ENVIO_AUTO] Nuevo estado leído:', nuevoEstado);
        addDebugLog('Nuevo estado: ' + (nuevoEstado ? 'Activado' : 'Desactivado'), 'info');
        
        var quiereActivar = nuevoEstado;
        var mensaje = quiereActivar
            ? '¿Desea activar el envío automático de documentos pendientes?'
            : '¿Desea desactivar el envío automático de documentos pendientes?';
        
        // Revertir el cambio inmediatamente hasta que el usuario confirme
        procesandoCambioEnvio = true;
        console.log('↩️ [ENVIO_AUTO] Revirtiendo cambio visual...');
        addDebugLog('Revirtiendo cambio visual', 'info');
        $chk.prop('checked', !nuevoEstado);
        actualizarLabelEnvioAutomatico(!nuevoEstado);
        console.log('✅ [ENVIO_AUTO] Cambio revertido. Estado actual:', !nuevoEstado);
        addDebugLog('Cambio revertido. Estado: ' + (!nuevoEstado ? 'Activado' : 'Desactivado'), 'success');
        
        if (typeof Swal !== 'undefined') {
            console.log('💬 [ENVIO_AUTO] Mostrando modal SweetAlert...');
            addDebugLog('Mostrando modal de confirmación', 'info');
            Swal.fire({
                title: quiereActivar ? 'Activar envío automático' : 'Desactivar envío automático',
                text: mensaje,
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: 'Sí',
                cancelButtonText: 'Cancelar',
                confirmButtonColor: '#28a745',
                cancelButtonColor: '#6c757d'
            }).then(function(result) {
                console.log('📥 [ENVIO_AUTO] Respuesta del modal recibida');
                console.log('📥 [ENVIO_AUTO] result completo:', result);
                console.log('📥 [ENVIO_AUTO] result.isConfirmed:', result.isConfirmed);
                console.log('📥 [ENVIO_AUTO] result.value:', result.value);
                addDebugLog('Respuesta modal - isConfirmed: ' + result.isConfirmed + ', value: ' + result.value, 'response');
                
                procesandoCambioEnvio = false;
                
                // Compatibilidad con SweetAlert2 (isConfirmed) y SweetAlert antiguo (value === true)
                var confirmado = (result && result.isConfirmed === true) || (result && result.value === true);
                console.log('📊 [ENVIO_AUTO] Confirmado calculado:', confirmado);
                addDebugLog('Confirmado calculado: ' + confirmado, 'info');
                
                if (confirmado) {
                    console.log('✅ [ENVIO_AUTO] Usuario confirmó. Aplicando cambio...');
                    console.log('✅ [ENVIO_AUTO] Nuevo estado a aplicar:', nuevoEstado);
                    addDebugLog('Usuario confirmó. Aplicando cambio a: ' + (nuevoEstado ? 'Activado' : 'Desactivado'), 'success');
                    
                    // Aplicar el cambio confirmado
                    $chk.prop('checked', nuevoEstado);
                    console.log('✅ [ENVIO_AUTO] Checkbox actualizado a:', nuevoEstado);
                    addDebugLog('Checkbox actualizado', 'success');
                    
                    actualizarLabelEnvioAutomatico(nuevoEstado);
                    console.log('✅ [ENVIO_AUTO] Label actualizado');
                    addDebugLog('Label actualizado a: ' + (nuevoEstado ? 'Activado' : 'Desactivado'), 'success');
                    
                    $('#btn_guardar_envio_automatico').show();
                    console.log('✅ [ENVIO_AUTO] Botón guardar mostrado');
                    addDebugLog('Botón guardar mostrado', 'success');
                    
                    console.log('✅ [ENVIO_AUTO] Cambio completado exitosamente');
                    addDebugLog('Cambio completado exitosamente', 'success');
                } else {
                    console.log('❌ [ENVIO_AUTO] Usuario canceló');
                    addDebugLog('Usuario canceló el cambio', 'warning');
                    // Ya está revertido, no hacer nada más
                }
            }).catch(function(error) {
                console.error('❌ [ENVIO_AUTO] Error en promesa de Swal:', error);
                addDebugLog('Error en promesa Swal: ' + error, 'error');
                procesandoCambioEnvio = false;
            });
        } else {
            console.log('💬 [ENVIO_AUTO] Usando confirm nativo...');
            addDebugLog('Usando confirm nativo (SweetAlert no disponible)', 'info');
            if (confirm(mensaje)) {
                procesandoCambioEnvio = false;
                console.log('✅ [ENVIO_AUTO] Usuario confirmó (confirm nativo)');
                addDebugLog('Usuario confirmó (confirm nativo)', 'success');
                // Aplicar el cambio confirmado
                $chk.prop('checked', nuevoEstado);
                actualizarLabelEnvioAutomatico(nuevoEstado);
                $('#btn_guardar_envio_automatico').show();
            } else {
                procesandoCambioEnvio = false;
                console.log('❌ [ENVIO_AUTO] Usuario canceló (confirm nativo)');
                addDebugLog('Usuario canceló (confirm nativo)', 'warning');
                // Ya está revertido, no hacer nada más
            }
        }
    });
    $('#btn_guardar_envio_automatico').on('click', function() {
        var valor = $('#switch_envio_automatico').is(':checked') ? 1 : 0;
        var $btn = $(this);
        $btn.prop('disabled', true);
        $.ajax({
            url: $('#url').val() + 'facturacion/documentos_guardar_envio_automatico',
            type: 'POST',
            data: { envio_automatico: valor },
            dataType: 'json',
            success: function(res) {
                $btn.prop('disabled', false);
                if (res && res.success) {
                    $('#btn_guardar_envio_automatico').hide();
                    actualizarLabelEnvioAutomatico(valor === 1);
                    if (typeof Swal !== 'undefined') {
                        Swal.fire({ title: 'Guardado', text: res.message || 'Configuración guardada.', icon: 'success', timer: 2000, showConfirmButton: false });
                    } else {
                        alert(res.message || 'Configuración guardada.');
                    }
                } else {
                    if (typeof Swal !== 'undefined') {
                        Swal.fire('Error', (res && res.message) ? res.message : 'No se pudo guardar.', 'error');
                    } else {
                        alert((res && res.message) ? res.message : 'No se pudo guardar.');
                    }
                }
            },
            error: function() {
                $btn.prop('disabled', false);
                if (typeof Swal !== 'undefined') {
                    Swal.fire('Error', 'Error de conexión al guardar.', 'error');
                } else {
                    alert('Error de conexión al guardar.');
                }
            }
        });
    });
    
    // Inicializar datepickers usando bootstrap-datepicker
    $('#filtro_fecha_desde').datepicker({
        format: 'yyyy-mm-dd',
        autoclose: true,
        todayHighlight: true
    });
    
    $('#filtro_fecha_hasta').datepicker({
        format: 'yyyy-mm-dd',
        autoclose: true,
        todayHighlight: true
    });
    
    // Establecer fechas por defecto (últimos 30 días)
    var fechaHasta = new Date();
    var fechaDesde = new Date();
    fechaDesde.setDate(fechaDesde.getDate() - 30);
    
    $('#filtro_fecha_desde').val(fechaDesde.toISOString().split('T')[0]);
    $('#filtro_fecha_hasta').val(fechaHasta.toISOString().split('T')[0]);
    
    console.log('📅 Fechas configuradas - Desde:', $('#filtro_fecha_desde').val(), 'Hasta:', $('#filtro_fecha_hasta').val());
    addDebugLog('Fechas configuradas - Desde: ' + $('#filtro_fecha_desde').val() + ' Hasta: ' + $('#filtro_fecha_hasta').val(), 'info');
    
    // Inicializar DataTable
    var table = $('#table-documentos-fe').DataTable({
        "processing": true,
        "serverSide": true,
        "ajax": {
            "url": $('#url').val() + 'facturacion/documentos_list',
            "type": "POST",
            "data": function(d) {
                var requestData = {
                    draw: d.draw,
                    start: d.start,
                    length: d.length,
                    fecha_desde: $('#filtro_fecha_desde').val(),
                    fecha_hasta: $('#filtro_fecha_hasta').val(),
                    status: $('#filtro_status').val(),
                    tipo_documento: $('#filtro_tipo_documento').val()
                };
                console.log('📤 Enviando petición AJAX:', requestData);
                console.log('🔗 URL:', $('#url').val() + 'facturacion/documentos_list');
                addDebugLog('Enviando petición - Desde: ' + requestData.fecha_desde + ', Hasta: ' + requestData.fecha_hasta, 'request');
                return requestData;
            },
            "dataSrc": function(json) {
                console.log('📥 Respuesta recibida del servidor:');
                console.log('  - Draw:', json.draw);
                console.log('  - RecordsTotal:', json.recordsTotal);
                console.log('  - RecordsFiltered:', json.recordsFiltered);
                console.log('  - Data length:', json.data ? json.data.length : 0);
                console.log('  - Datos completos:', json);
                addDebugLog('Respuesta recibida - Total: ' + json.recordsTotal + ', Filtrados: ' + json.recordsFiltered + ', Datos: ' + (json.data ? json.data.length : 0), json.data && json.data.length > 0 ? 'success' : 'warning');
                
                if (json.data && json.data.length > 0) {
                    console.log('✅ Documentos encontrados:', json.data.length);
                    addDebugLog('Documentos encontrados: ' + json.data.length, 'success');
                    json.data.forEach(function(doc, index) {
                        console.log('  Documento ' + (index + 1) + ':', {
                            id: doc.id_documento_fe,
                            tipo: doc.document_type,
                            serie: doc.serie_electronica,
                            numero: doc.fd_number,
                            cliente: doc.receptor_name,
                            total: doc.total,
                            estado: doc.status
                        });
                        addDebugLog('Doc ' + (index + 1) + ': ID=' + doc.id_documento_fe + ', Tipo=' + doc.document_type + ', Serie=' + doc.serie_electronica, 'info');
                    });
                } else {
                    console.warn('⚠️ No se encontraron documentos en la respuesta');
                    console.warn('⚠️ JSON completo:', JSON.stringify(json, null, 2));
                    addDebugLog('No se encontraron documentos en la respuesta. Verifique los filtros de fecha.', 'warning');
                }
                
                return json.data || [];
            },
            "error": function(xhr, error, thrown) {
                console.error('❌ Error en DataTable:');
                console.error('  - Error:', error);
                console.error('  - Thrown:', thrown);
                console.error('  - Status:', xhr.status);
                console.error('  - StatusText:', xhr.statusText);
                console.error('  - Response Text:', xhr.responseText);
                console.error('  - Response Headers:', xhr.getAllResponseHeaders());
                addDebugLog('Error: ' + error + ' (Status: ' + xhr.status + ')', 'error');
                
                // Intentar parsear la respuesta como JSON
                try {
                    var jsonResponse = JSON.parse(xhr.responseText);
                    console.error('  - JSON Response:', jsonResponse);
                    addDebugLog('Respuesta de error: ' + JSON.stringify(jsonResponse), 'error');
                } catch (e) {
                    console.error('  - No se pudo parsear como JSON');
                    addDebugLog('No se pudo parsear respuesta como JSON: ' + xhr.responseText.substring(0, 200), 'error');
                }
            }
        },
        "columns": [
            { "data": "id_documento_fe" },
            { 
                "data": "fecha_creacion",
                "render": function(data) {
                    if (data) {
                        var fecha = new Date(data);
                        return fecha.toLocaleDateString('es-PA') + ' ' + fecha.toLocaleTimeString('es-PA');
                    }
                    return '-';
                }
            },
            { 
                "data": "document_type",
                "render": function(data) {
                    var tipos = {
                        'FAC': '<span class="label label-info">Factura</span>',
                        'NTC': '<span class="label label-warning">Nota Crédito</span>',
                        'NTD': '<span class="label label-danger">Nota Débito</span>'
                    };
                    return tipos[data] || data;
                }
            },
            { "data": "serie_electronica" },
            { 
                "data": "fd_number",
                "render": function(data) {
                    return String(data).padStart(10, '0');
                }
            },
            { "data": "receptor_name" },
            { 
                "data": "total",
                "render": function(data) {
                    return parseFloat(data).toFixed(2);
                },
                "className": "text-right"
            },
            { 
                "data": "status",
                "render": function(data, type, row) {
                    var estados = {
                        'pending': '<span class="label label-warning">Pendiente</span>',
                        'sent': '<span class="label label-info">Enviado</span>',
                        'authorized': '<span class="label label-success">Autorizado</span>',
                        'rejected': '<span class="label label-danger">Rechazado</span>'
                    };
                    var html = estados[data] || data;
                    if (row.rejected == 1 || data === 'rejected') {
                        html += ' <button type="button" class="btn btn-link btn-sm p-0 btn-ver-motivo-rechazo" data-id="' + row.id_documento_fe + '" title="Ver motivo del rechazo"><i class="fa fa-exclamation-circle text-danger"></i> Ver motivo</button>';
                    }
                    return html;
                }
            },
            { 
                "data": "ntc_emitida",
                "render": function(data, type, row) {
                    if (!data) return '-';
                    return '<span class="text-success" title="Nota de crédito emitida para esta factura"><i class="fa fa-file-invoice-dollar"></i> ' + data + '</span>';
                },
                "orderable": false
            },
            { 
                "data": "cufe",
                "render": function(data) {
                    return data ? '<small>' + data.substring(0, 20) + '...</small>' : '-';
                }
            },
            { 
                "data": null,
                "defaultContent": '',
                "orderable": false,
                "render": function(data, type, row) {
                    var html = '<div class="btn-group">';
                    
                    // Botón Ver
                    html += '<button class="btn btn-info btn-sm btn-ver" title="Ver detalles" data-id="' + row.id_documento_fe + '"><i class="fa fa-eye"></i></button>';
                    
                    // Botón Ver comprobante (formato Panamá - ticket con CUFE y QR)
                    html += '<a href="' + ($('#url').val()) + 'facturacion/documentos_comprobante_panama?id=' + row.id_documento_fe + '" target="_blank" class="btn btn-secondary btn-sm" title="Ver comprobante (formato Panamá)"><i class="fa fa-receipt"></i> Comprobante</a>';
                    
                    // Botón Convertir (solo para NOTA VENTA - id_tipo_doc_venta = 3)
                    if (row.id_tipo_doc_venta == 3 || row.id_tipo_doc_venta == '3') {
                        html += '<button class="btn btn-sm btn-convertir" style="background-color: #ff6b6b; border-color: #ff6b6b; color: white;" title="Convertir a Boleta o Factura" data-id="' + row.id_documento_fe + '" data-id-venta="' + (row.id_venta || '') + '" data-cliente-actual="' + (row.receptor_name || '') + '" data-cliente-id="' + (row.id_cliente || '') + '"><i class="fa fa-exchange-alt"></i> Convertir</button>';
                    }
                    
                    // Botón Emitir Nota de Crédito (solo para FACTURAS autorizadas - document_type = FAC y status = authorized)
                    if (row.document_type == 'FAC' && (row.status == 'authorized' || row.status == 'sent')) {
                        html += '<button class="btn btn-warning btn-sm btn-emitir-nota-credito" title="Emitir Nota de Crédito" data-id="' + row.id_documento_fe + '" data-fd-number="' + (row.fd_number || '') + '" data-fecha="' + (row.fecha_creacion || '') + '"><i class="fa fa-file-invoice-dollar"></i> N. Crédito</button>';
                    }
                    
                    // Botón Enviar (solo si está pendiente)
                    if (row.status == 'pending') {
                        html += '<button class="btn btn-success btn-sm btn-enviar" title="Enviar a facturafacil" data-id="' + row.id_documento_fe + '"><i class="fa fa-paper-plane"></i></button>';
                    }
                    
                    // Botón Reenviar (si ya fue enviado pero puede reenviarse)
                    if (row.status == 'sent' || row.status == 'authorized' || row.status == 'rejected' || (row.status == 'pending' && row.fecha_envio)) {
                        html += '<button class="btn btn-warning btn-sm btn-reenviar" title="Reenviar a facturafacil" data-id="' + row.id_documento_fe + '"><i class="fa fa-redo"></i></button>';
                    }
                    
                    // Botón Descargar PDF (solo si tiene PDF)
                    if (row.pdf_url) {
                        html += '<button class="btn btn-danger btn-sm btn-descargar-pdf" title="Descargar PDF" data-id="' + row.id_documento_fe + '"><i class="fa fa-file-pdf"></i></button>';
                    }
                    
                    // Botón Descargar CUFE (solo si tiene CUFE y está autorizado)
                    if (row.cufe && row.document_uuid) {
                        html += '<button class="btn btn-info btn-sm btn-descargar-cafe" title="Descargar CUFE" data-id="' + row.id_documento_fe + '" data-cufe="' + (row.cufe || '') + '"><i class="fa fa-file-invoice"></i> CUFE</button>';
                    }
                    
                    // Botón Descargar XML (solo si tiene XML)
                    if (row.xml_data) {
                        html += '<button class="btn btn-warning btn-sm btn-descargar-xml" title="Descargar XML" data-id="' + row.id_documento_fe + '"><i class="fa fa-file-code"></i></button>';
                    }
                    
                    // Botón Enviar Email (solo si tiene email del receptor)
                    if (row.receptor_email) {
                        html += '<button class="btn btn-primary btn-sm btn-enviar-email" title="Enviar por correo" data-id="' + row.id_documento_fe + '"><i class="fa fa-envelope"></i></button>';
                    }
                    
                    html += '</div>';
                    return html;
                }
            }
        ],
        "language": {
            "url": "//cdn.datatables.net/plug-ins/1.10.24/i18n/Spanish.json"
        },
        "order": [[0, "desc"]],
        "initComplete": function(settings, json) {
            console.log('✅ DataTable inicializado correctamente');
            console.log('📊 Total de registros:', json.recordsTotal);
            console.log('🔍 Registros filtrados:', json.recordsFiltered);
        }
    });
    
    console.log('📋 DataTable configurado, esperando datos...');
    
    // Aplicar filtros
    $('#filtro_fecha_desde, #filtro_fecha_hasta, #filtro_status, #filtro_tipo_documento').on('change', function() {
        console.log('🔄 Recargando tabla con nuevos filtros...');
        table.ajax.reload();
    });
    
    // Ver detalles del documento
    $('#table-documentos-fe tbody').on('click', '.btn-ver', function() {
        var id = $(this).data('id');
        verDocumento(id);
    });
    // Ver motivo de rechazo (abre el mismo modal de detalles con la sección de mensajes)
    $('#table-documentos-fe tbody').on('click', '.btn-ver-motivo-rechazo', function(e) {
        e.preventDefault();
        var id = $(this).data('id');
        verDocumento(id);
    });
    
    // Convertir documento (NOTA VENTA a Boleta/Factura)
    $('#table-documentos-fe tbody').on('click', '.btn-convertir', function() {
        var idDocumento = $(this).data('id');
        var idVenta = $(this).data('id-venta');
        var clienteActual = $(this).data('cliente-actual') || '';
        var clienteIdActual = $(this).data('cliente-id') || '';
        
        console.log('🔄 Botón convertir clickeado, ID Documento:', idDocumento, 'ID Venta:', idVenta);
        addDebugLog('Botón convertir clickeado para documento ID: ' + idDocumento, 'info');
        
        $('#id_documento_convertir').val(idDocumento);
        $('#id_venta_convertir').val(idVenta);
        $('#cliente_id_original_convertir').val(clienteIdActual);
        
        // Cargar datos del documento para obtener información del cliente
        $.ajax({
            url: $('#url').val() + 'facturacion/documentos_data',
            type: 'POST',
            dataType: 'json',
            data: { id_documento_fe: idDocumento },
            success: function(response) {
                if (response && response.success && response.data) {
                    var docData = response.data;
                    
                    // Establecer cliente actual
                    if (docData.id_cliente) {
                        $('#cliente_id_convertir').val(docData.id_cliente);
                        $('#cliente_id_original_convertir').val(docData.id_cliente);
                    }
                    if (docData.receptor_name) {
                        $('#cliente-nombre-actual-convertir').text(docData.receptor_name);
                        $('#buscar_cliente_convertir').val(docData.receptor_name);
                    }
                    
                    // Actualizar opción de cliente
                    if (docData.id_cliente && docData.id_cliente != '1') {
                        $('.opcion-cliente-convertir').html('<a class="input-group-prepend" href="javascript:void(0)" onclick="editarClienteConvertir(' + docData.id_cliente + ');" data-original-title="Editar cliente" data-toggle="tooltip" data-placement="top">' +
                            '<span class="input-group-text bg-header">' +
                                '<small><i class="fas fa-user text-info"></i></small>' +
                            '</span>' +
                        '</a>');
                    }
                }
            },
            error: function() {
                // Si hay error, usar datos del botón
                if (clienteActual) {
                    $('#cliente-nombre-actual-convertir').text(clienteActual);
                    $('#buscar_cliente_convertir').val(clienteActual);
                }
            }
        });
        
        // Resetear selección a BOLETA por defecto
        $('input:radio[name="tipo_doc_convertir"][value="1"]').prop('checked', true);
        $('.btn-tipo-convertir-1').addClass('active');
        $('.btn-tipo-convertir-2').removeClass('active');
        
        // Mostrar campo de cliente
        $('#div-cliente-convertir').show();
        
        $('#modal-convertir-documento').modal('show');
    });
    
    // Inicializar autocomplete cuando el modal se muestra completamente
    $('#modal-convertir-documento').on('shown.bs.modal', function() {
        setTimeout(function() {
            inicializarAutocompleteCliente();
        }, 100);
    });
    
    // Función para inicializar autocomplete de cliente
    function inicializarAutocompleteCliente() {
        // Verificar que el elemento existe
        if ($("#buscar_cliente_convertir").length === 0) {
            console.error('Elemento #buscar_cliente_convertir no encontrado');
            return;
        }
        
        // Destruir autocomplete existente si existe
        try {
            if ($("#buscar_cliente_convertir").autocomplete("instance")) {
                $("#buscar_cliente_convertir").autocomplete("destroy");
            }
        } catch(e) {
            // Ignorar errores al destruir
        }
        
        // Inicializar autocomplete para búsqueda de cliente en modal de conversión
        $("#buscar_cliente_convertir").autocomplete({
            delay: 1,
            autoFocus: true,
            source: function (request, response) {
                $.ajax({
                    url: $('#url').val()+'venta/buscar_cliente',
                    type: "post",
                    dataType: "json",
                    data: {
                        cadena: request.term,
                        tipo_cliente: $('#cliente_tipo_convertir').val() || '1'
                    },
                    success: function (data) {
                        if (data && Array.isArray(data)) {
                            response($.map(data, function (item) {
                                var tipo_cli = (item.tipo_cliente == 1) ? "DNI" : "RUC";
                                var identificacion = item.tipo_cliente == 1 ? item.dni : item.ruc;
                                return {
                                    id: item.id_cliente,
                                    dni: item.dni,
                                    ruc: item.ruc,
                                    tipo: item.tipo_cliente,
                                    nombres: item.nombre,
                                    fecha_n: item.fecha_nac,
                                    label: tipo_cli+': '+identificacion+' | '+item.nombre,
                                    value: tipo_cli+': '+identificacion+' | '+item.nombre
                                }
                            }));
                        } else {
                            response([]);
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('Error en autocomplete:', error);
                        response([]);
                    }
                })
            },
            select: function (e, ui) {
                $("#cliente_id_convertir").val(ui.item.id);
                $('#cliente_tipo_convertir').val(ui.item.tipo);
                $(this).blur();
                $('.opcion-cliente-convertir').html('<a class="input-group-prepend" href="javascript:void(0)"'
                    +'onclick="editarClienteConvertir('+ui.item.id+');" data-original-title="Editar cliente" data-toggle="tooltip"'
                    +'data-placement="top">'
                        +'<span class="input-group-text bg-header">'
                            +'<small><i class="fas fa-user text-info"></i></small>'
                       +'</span>'
                    +'</a>');
                $('#cliente-nombre-actual-convertir').text(ui.item.nombres);
            }
        });
        $("#buscar_cliente_convertir").autocomplete("option", "appendTo", "#modal-convertir-documento");
    }
    
    // Limpiar cliente en modal de conversión
    $("#btnClienteLimpiarConvertir").click(function() {
        $("#cliente_id_convertir").val('');
        $("#buscar_cliente_convertir").val('');
        $("#buscar_cliente_convertir").focus();
        $('.opcion-cliente-convertir').html('<a class="input-group-prepend" href="javascript:void(0)"'
            +'onclick="nuevoClienteConvertir();" data-original-title="Registrar nuevo cliente" data-toggle="tooltip"'
            +'data-placement="top">'
                +'<span class="input-group-text bg-header">'
                    +'<small><i class="fas fa-user-plus"></i></small>'
               +'</span>'
            +'</a>');
        $('#cliente-nombre-actual-convertir').text('-');
    });
    
    // Función helper para abrir el modal de cliente manualmente
    function abrirModalClienteManual() {
        console.log('🔄 abrirModalClienteManual() llamado');
        addDebugLog('Abriendo modal manualmente', 'info');
        
        if ($('#modal-cliente').length === 0) {
            console.log('❌ Modal no existe en el DOM');
            addDebugLog('Modal no existe en el DOM', 'error');
            return;
        }
        
        $('#id_cliente').val('');
        $('#tipo_cliente').val(1);
        $('.modal-title-cliente').text('Nuevo Cliente');
        
        // Configurar campos según tipo DNI por defecto
        $("#td_dni").prop('checked', true);
        $("#td_ruc").prop('checked', false);
        $(".block_ce").css("display","none");
        $(".block01").css("display","block");
        $(".block02").css("display","none");
        $(".block03").css("display","block");
        $(".block04").css("display","block");
        $(".block05").css("display","block");
        $(".block06").css("display","block");
        $(".block07").css("display","none");
        $(".dni").prop('disabled', false);
        $(".ruc").prop('disabled', true);
        
        // Limpiar formulario
        if ($('#form-cliente').length > 0) {
            $('#form-cliente')[0].reset();
        }
        $('#id_cliente').val('');
        $('#tipo_cliente').val(1);
        
        console.log('✅ Abriendo modal...');
        addDebugLog('Abriendo modal de cliente', 'success');
        $('#modal-cliente').modal('show');
        
        // Verificar si se abrió
        setTimeout(function() {
            if ($('#modal-cliente').hasClass('show') || $('#modal-cliente').is(':visible')) {
                console.log('✅ Modal abierto correctamente');
                addDebugLog('Modal abierto correctamente', 'success');
            } else {
                console.log('❌ Modal no se abrió');
                console.log('  - Modal existe:', $('#modal-cliente').length > 0);
                console.log('  - Modal tiene clase show:', $('#modal-cliente').hasClass('show'));
                console.log('  - Modal visible:', $('#modal-cliente').is(':visible'));
                addDebugLog('Modal no se abrió', 'error');
            }
        }, 500);
    }
    
    // Funciones helper para el modal de conversión
    window.nuevoClienteConvertir = function() {
        console.log('🔵 nuevoClienteConvertir llamado');
        addDebugLog('nuevoClienteConvertir llamado', 'info');
        
        // Verificar funciones disponibles
        console.log('🔍 Verificando funciones disponibles:');
        console.log('  - typeof nuevoCliente:', typeof nuevoCliente);
        console.log('  - typeof nuevo_cliente:', typeof nuevo_cliente);
        console.log('  - typeof nuevo:', typeof nuevo);
        console.log('  - Modal existe:', $('#modal-cliente').length > 0);
        
        // Intentar usar funciones disponibles primero (si están cargadas desde venta_cliente.js)
        // Preferir nuevo_cliente() porque es más directo y no depende de campos específicos
        if (typeof nuevo_cliente === 'function') {
            console.log('✅ Usando función nuevo_cliente()');
            addDebugLog('Usando función nuevo_cliente()', 'success');
            try {
                nuevo_cliente();
                console.log('✅ nuevo_cliente() ejecutado');
                addDebugLog('nuevo_cliente() ejecutado', 'success');
                
                // Verificar si el modal se abrió después de un tiempo
                setTimeout(function() {
                    if ($('#modal-cliente').hasClass('show') || $('#modal-cliente').is(':visible')) {
                        console.log('✅ Modal abierto después de nuevo_cliente()');
                        addDebugLog('Modal abierto después de nuevo_cliente()', 'success');
                    } else {
                        console.log('❌ Modal NO abierto después de nuevo_cliente()');
                        console.log('  - Modal existe:', $('#modal-cliente').length > 0);
                        console.log('  - Modal tiene clase show:', $('#modal-cliente').hasClass('show'));
                        console.log('  - Modal visible:', $('#modal-cliente').is(':visible'));
                        addDebugLog('Modal NO abierto después de nuevo_cliente()', 'error');
                        
                        // Intentar abrir manualmente
                        console.log('🔄 Intentando abrir modal manualmente...');
                        abrirModalClienteManual();
                    }
                }, 500);
            } catch(error) {
                console.error('❌ Error al ejecutar nuevo_cliente():', error);
                addDebugLog('Error al ejecutar nuevo_cliente(): ' + error.message, 'error');
                abrirModalClienteManual();
            }
            return;
        }
        if (typeof nuevoCliente === 'function') {
            console.log('✅ Usando función nuevoCliente()');
            addDebugLog('Usando función nuevoCliente()', 'success');
            try {
                nuevoCliente();
                console.log('✅ nuevoCliente() ejecutado');
                addDebugLog('nuevoCliente() ejecutado', 'success');
                
                // Verificar si el modal se abrió después de un tiempo
                setTimeout(function() {
                    if ($('#modal-cliente').hasClass('show') || $('#modal-cliente').is(':visible')) {
                        console.log('✅ Modal abierto después de nuevoCliente()');
                        addDebugLog('Modal abierto después de nuevoCliente()', 'success');
                    } else {
                        console.log('❌ Modal NO abierto después de nuevoCliente()');
                        addDebugLog('Modal NO abierto después de nuevoCliente()', 'error');
                        abrirModalClienteManual();
                    }
                }, 500);
            } catch(error) {
                console.error('❌ Error al ejecutar nuevoCliente():', error);
                addDebugLog('Error al ejecutar nuevoCliente(): ' + error.message, 'error');
                abrirModalClienteManual();
            }
            return;
        }
        if (typeof nuevo === 'function') {
            console.log('✅ Usando función nuevo()');
            addDebugLog('Usando función nuevo()', 'success');
            nuevo();
            return;
        }
        
        // Si el modal existe en el DOM, inicializarlo y abrirlo directamente
        if ($('#modal-cliente').length > 0) {
            console.log('✅ Modal encontrado, inicializando manualmente...');
            addDebugLog('Modal encontrado, inicializando manualmente...', 'info');
            abrirModalClienteManual();
            return;
        }
        
        // Si no hay modal, mostrar mensaje
        console.log('❌ Modal no encontrado en el DOM');
        addDebugLog('Modal no encontrado en el DOM', 'error');
        Swal.fire({
            title: 'Error',
            text: 'No se pudo cargar el formulario de cliente. Por favor, recargue la página.',
            icon: 'error',
            confirmButtonText: 'Aceptar'
        });
    };
    
    window.editarClienteConvertir = function(id_cliente) {
        if (!id_cliente || id_cliente == '' || id_cliente == '1') {
            nuevoClienteConvertir();
            return;
        }
        
        // Verificar si el modal de cliente existe en el DOM
        if ($('#modal-cliente').length > 0) {
            // Si existe, usar las funciones disponibles
            if (typeof editar_cliente === 'function') {
                editar_cliente(id_cliente);
            } else {
                // Cargar datos del cliente y abrir modal
                $.ajax({
                    url: $('#url').val() + 'cliente/cliente_datos',
                    type: 'POST',
                    data: { id_cliente: id_cliente },
                    dataType: 'json',
                    success: function(response) {
                        if (response && response.success && response.data && response.data.length > 0) {
                            var cliente = response.data[0];
                            $('#id_cliente').val(cliente.id_cliente);
                            $('#tipo_cliente').val(cliente.tipo_cliente);
                            $('.modal-title-cliente').text('Editar Cliente');
                            $('#modal-cliente').modal('show');
                        }
                    },
                    error: function() {
                        Swal.fire({
                            title: 'Error',
                            text: 'No se pudieron cargar los datos del cliente',
                            icon: 'error',
                            confirmButtonText: 'Aceptar'
                        });
                    }
                });
            }
        } else {
            // Cargar el modal de cliente desde la vista de venta
            $.ajax({
                url: $('#url').val() + 'venta/orden',
                type: 'GET',
                success: function(html) {
                    var modalHtml = $(html).find('#modal-cliente').parent().html();
                    if (modalHtml) {
                        $('body').append(modalHtml);
                        // Cargar datos del cliente
                        $.ajax({
                            url: $('#url').val() + 'cliente/cliente_datos',
                            type: 'POST',
                            data: { id_cliente: id_cliente },
                            dataType: 'json',
                            success: function(response) {
                                if (response && response.success && response.data && response.data.length > 0) {
                                    var cliente = response.data[0];
                                    $('#id_cliente').val(cliente.id_cliente);
                                    $('#tipo_cliente').val(cliente.tipo_cliente);
                                    $('.modal-title-cliente').text('Editar Cliente');
                                    $('#modal-cliente').modal('show');
                                }
                            }
                        });
                    }
                }
            });
        }
    };
    
    // Listener para cuando se cierra el modal de cliente y se actualiza el cliente
    // Esto actualiza el campo de búsqueda en el modal de conversión si está abierto
    $('#modal-cliente').on('hidden.bs.modal', function() {
        // Verificar si el modal de conversión está abierto
        if ($('#modal-convertir-documento').hasClass('show') || $('#modal-convertir-documento').is(':visible')) {
            // Obtener el cliente que se acaba de guardar/editar
            var clienteId = $('#id_cliente').val() || $('#cliente_id').val();
            if (clienteId && clienteId != '' && clienteId != '1') {
                // Cargar datos del cliente actualizado
                $.ajax({
                    url: $('#url').val() + 'cliente/cliente_datos',
                    type: 'POST',
                    data: { id_cliente: clienteId },
                    dataType: 'json',
                    success: function(response) {
                        if (response && response.success && response.data && response.data.length > 0) {
                            var cliente = response.data[0];
                            var tipo_cli = (cliente.tipo_cliente == 1) ? "DNI" : "RUC";
                            var nombreCliente = cliente.nombres || cliente.razon_social || '';
                            
                            $('#cliente_id_convertir').val(cliente.id_cliente);
                            $('#cliente_tipo_convertir').val(cliente.tipo_cliente);
                            $('#buscar_cliente_convertir').val(tipo_cli + ': ' + (cliente.dni || '') + (cliente.ruc || '') + ' | ' + nombreCliente);
                            $('#cliente-nombre-actual-convertir').text(nombreCliente);
                            
                            $('.opcion-cliente-convertir').html('<a class="input-group-prepend" href="javascript:void(0)" onclick="editarClienteConvertir(' + cliente.id_cliente + ');" data-original-title="Editar cliente" data-toggle="tooltip" data-placement="top">' +
                                '<span class="input-group-text bg-header">' +
                                    '<small><i class="fas fa-user text-info"></i></small>' +
                                '</span>' +
                            '</a>');
                        }
                    }
                });
            }
        }
    });
    
    // Manejar cambio de tipo de documento en modal de conversión
    $('input[name="tipo_doc_convertir"]').on('change', function(){
        var tipoDoc = $(this).val();
        $('.btn-tipo-convertir-1, .btn-tipo-convertir-2').removeClass('active');
        $('.btn-tipo-convertir-' + tipoDoc).addClass('active');
        
        // Si es Factura (tipo 2), puede requerir validación de cliente con RUC
        // Por ahora solo mostramos el campo siempre
    });
    
    // Confirmar conversión
    $('#btn-confirmar-conversion').on('click', function() {
        var idDocumento = $('#id_documento_convertir').val();
        var idVenta = $('#id_venta_convertir').val();
        var tipoDocNuevo = $('input:radio[name="tipo_doc_convertir"]:checked').val();
        var tipoDocNombre = tipoDocNuevo == '1' ? 'BOLETA' : 'FACTURA';
        var clienteIdNuevo = $('#cliente_id_convertir').val() || $('#cliente_id_original_convertir').val();
        
        console.log('🔄 Confirmando conversión:', {
            idDocumento: idDocumento,
            idVenta: idVenta,
            tipoDocNuevo: tipoDocNuevo,
            tipoDocNombre: tipoDocNombre,
            clienteIdNuevo: clienteIdNuevo
        });
        
        if (!idDocumento || !idVenta || !tipoDocNuevo) {
            Swal.fire('Error', 'Faltan datos para realizar la conversión', 'error');
            return;
        }
        
        // Deshabilitar botón mientras se procesa
        $(this).prop('disabled', true).text('Convirtiendo...');
        
        $.ajax({
            url: $('#url').val() + 'facturacion/convertir_documento',
            type: 'POST',
            dataType: 'json',
            data: {
                id_documento_fe: idDocumento,
                id_venta: idVenta,
                nuevo_tipo_doc: tipoDocNuevo,
                nuevo_cliente_id: clienteIdNuevo // Cliente seleccionado (si se cambió)
            },
            success: function(response) {
                console.log('✅ Respuesta de conversión:', response);
                addDebugLog('Conversión completada: ' + JSON.stringify(response), 'success');
                
                if (response.success) {
                    Swal.fire({
                        title: '¡Conversión Exitosa!',
                        html: 'La Nota de Venta ha sido convertida a <strong>' + tipoDocNombre + '</strong> exitosamente.<br><br>' +
                              (response.id_documento_nuevo ? 'Nuevo documento ID: ' + response.id_documento_nuevo : ''),
                        icon: 'success',
                        confirmButtonText: 'Aceptar',
                        allowOutsideClick: false
                    }).then(function() {
                        // Recargar tabla
                        table.ajax.reload();
                        $('#modal-convertir-documento').modal('hide');
                    });
                } else {
                    Swal.fire('Error', response.message || 'Error al convertir el documento', 'error');
                    $('#btn-confirmar-conversion').prop('disabled', false).text('Convertir');
                }
            },
            error: function(xhr, status, error) {
                console.error('❌ Error en conversión:', error);
                console.error('❌ Response:', xhr.responseText);
                addDebugLog('Error en conversión: ' + error, 'error');
                Swal.fire('Error', 'Error al procesar la conversión. Por favor, intente nuevamente.', 'error');
                $('#btn-confirmar-conversion').prop('disabled', false).text('Convertir');
            }
        });
    });
    
    // Enviar documento
    $('#table-documentos-fe tbody').on('click', '.btn-enviar', function() {
        var id = $(this).data('id');
        $('#id_documento_enviar').val(id);
        $('#modal-enviar-documento').modal('show');
    });
    
    // Reenviar documento (usa la misma función de envío)
    $('#table-documentos-fe tbody').on('click', '.btn-reenviar', function() {
        var id = $(this).data('id');
        console.log('🔄 Botón reenviar clickeado, ID:', id);
        addDebugLog('Botón reenviar clickeado para documento ID: ' + id, 'info');
        
        // Validar que el ID existe
        if (!id || id === 'undefined' || id === '') {
            console.error('❌ ID inválido en botón reenviar:', id);
            Swal.fire('Error', 'ID de documento inválido', 'error');
            return;
        }
        
        // Usar la misma función de envío pero con confirmación de reenvío
        Swal.fire({
            title: '¿Reenviar documento?',
            text: 'Este documento será reenviado a facturafacil. ¿Desea continuar?',
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: 'Sí, reenviar',
            cancelButtonText: 'Cancelar',
            confirmButtonColor: '#ffc107',
            allowOutsideClick: false
        }).then((result) => {
            console.log('🔄 Resultado de confirmación reenvío:', result);
            console.log('🔄 result.isConfirmed:', result.isConfirmed);
            console.log('🔄 result.value:', result.value);
            
            if (result.isConfirmed || result.value === true) {
                console.log('✅ Confirmado, llamando a enviarDocumento con ID:', id);
                addDebugLog('Confirmado reenvío, iniciando envío del documento ID: ' + id, 'success');
                // Pequeño delay para asegurar que el modal se cierre completamente
                setTimeout(function() {
                    enviarDocumento(id);
                }, 100);
            } else {
                console.log('❌ Reenvío cancelado');
                addDebugLog('Reenvío cancelado por el usuario', 'warning');
            }
        }).catch((error) => {
            console.error('❌ Error en promesa de Swal:', error);
            addDebugLog('Error en confirmación de reenvío: ' + error, 'error');
        });
    });
    
    // Confirmar envío
    $('#btn-confirmar-envio').on('click', function() {
        var id = $('#id_documento_enviar').val();
        enviarDocumento(id);
    });
    
    // Descargar PDF
    $('#table-documentos-fe tbody').on('click', '.btn-descargar-pdf', function() {
        var id = $(this).data('id');
        window.open($('#url').val() + 'facturacion/documentos_descargar_pdf?id=' + id, '_blank');
    });
    
    // Descargar CUFE
    $('#table-documentos-fe tbody').on('click', '.btn-descargar-cafe', function() {
        var id = $(this).data('id');
        var cufe = $(this).data('cufe');
        console.log('Descargando CUFE para documento ID:', id, 'CUFE:', cufe);
        
        // Abrir en nueva ventana para descargar el CUFE
        // Por defecto: 1 copia, descarga (no view)
        window.open($('#url').val() + 'facturacion/documentos_descargar_cafe?id=' + id + '&copies=1&view=false', '_blank');
    });
    
    // Descargar XML
    $('#table-documentos-fe tbody').on('click', '.btn-descargar-xml', function() {
        var id = $(this).data('id');
        window.open($('#url').val() + 'facturacion/documentos_descargar_xml?id=' + id, '_blank');
    });
    
    // Enviar por correo
    $('#table-documentos-fe tbody').on('click', '.btn-enviar-email', function() {
        var id = $(this).data('id');
        enviarPorEmail(id);
    });
    
    // Emitir Nota de Crédito
    $('#table-documentos-fe tbody').on('click', '.btn-emitir-nota-credito', function() {
        var idDocumento = $(this).data('id');
        var fdNumber = $(this).data('fd-number');
        var fecha = $(this).data('fecha');
        
        console.log('Emitir Nota de Crédito para documento:', idDocumento);
        addDebugLog('Abriendo modal para emitir Nota de Crédito', 'info');
        
        // Cargar datos del documento
        $.ajax({
            url: $('#url').val() + 'facturacion/documentos_data',
            type: 'POST',
            dataType: 'json',
            data: { id_documento_fe: idDocumento },
            success: function(response) {
                if (response && response.success && response.data) {
                    var doc = response.data;
                    
                    // Llenar datos en el modal
                    $('#nc-id-documento-original').val(idDocumento);
                    $('#nc-fd-number-original').val(doc.fd_number || fdNumber);
                    $('#nc-fecha-creacion-original').val(doc.fecha_creacion || fecha);
                    $('#nc-serie-original').text(doc.serie_electronica || '-');
                    $('#nc-numero-original').text(String(doc.fd_number || fdNumber).padStart(10, '0'));
                    $('#nc-fecha-original').text(doc.fecha_creacion || fecha);
                    $('#nc-cliente-original').text(doc.receptor_name || '-');
                    $('#nc-total-original').text('$' + parseFloat(doc.total || 0).toFixed(2));
                    $('#nc-cufe-original').text(doc.cufe || '-');
                    $('#nc-motivo').val('');
                    
                    $('#modal-emitir-nota-credito').modal('show');
                } else {
                    Swal.fire({
                        title: 'Error',
                        text: 'No se pudieron cargar los datos del documento',
                        icon: 'error',
                        confirmButtonText: 'Aceptar'
                    });
                }
            },
            error: function() {
                Swal.fire({
                    title: 'Error',
                    text: 'Error al cargar los datos del documento',
                    icon: 'error',
                    confirmButtonText: 'Aceptar'
                });
            }
        });
    });
    
    // Confirmar emisión de Nota de Crédito
    $('#btn-confirmar-emitir-nc').on('click', function() {
        var idDocumento = $('#nc-id-documento-original').val();
        var motivo = $('#nc-motivo').val();
        console.log('[NC] Click en Confirmar. id_documento_fe:', idDocumento, 'motivo:', motivo);
        addDebugLog('Click Confirmar NC - id_documento_fe: ' + idDocumento + ', motivo: ' + (motivo || '(vacío)'), 'info');
        
        if (!idDocumento) {
            console.log('[NC] Sin idDocumento, mostrando error');
            Swal.fire({
                title: 'Error',
                text: 'No se ha seleccionado un documento válido',
                icon: 'error',
                confirmButtonText: 'Aceptar'
            });
            return;
        }
        
        Swal.fire({
            title: '¿Confirmar emisión?',
            text: 'Se creará una Nota de Crédito que anulará la factura seleccionada',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#ffc107',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Sí, emitir',
            cancelButtonText: 'Cancelar'
        }).then((result) => {
            // Compatibilidad: SweetAlert2 usa result.isConfirmed; versiones antiguas pueden usar result.value === true
            var confirmado = !!(result && (result.isConfirmed === true || result.value === true));
            console.log('[NC] Resultado Swal completo:', result, 'confirmado=', confirmado);
            addDebugLog('Swal NC resultado: ' + JSON.stringify(result) + ' confirmado=' + confirmado, 'info');
            if (confirmado) {
                var urlNc = $('#url').val() + 'facturacion/emitir_nota_credito';
                console.log('[NC] Enviando POST a:', urlNc, 'data:', { id_documento_fe: idDocumento, motivo: motivo });
                addDebugLog('POST ' + urlNc + ' id_documento_fe=' + idDocumento, 'request');
                Swal.fire({
                    title: 'Procesando...',
                    text: 'Creando Nota de Crédito',
                    allowOutsideClick: false,
                    didOpen: () => {
                        Swal.showLoading();
                    }
                });
                
                $.ajax({
                    url: urlNc,
                    type: 'POST',
                    dataType: 'json',
                    data: {
                        id_documento_fe: idDocumento,
                        motivo: motivo
                    },
                    success: function(response) {
                        console.log('[NC] Respuesta success:', response);
                        addDebugLog('NC respuesta: ' + JSON.stringify(response), 'response');
                        Swal.close();
                        if (response && response.success) {
                            Swal.fire({
                                title: 'Éxito',
                                text: response.message || 'Nota de Crédito emitida correctamente',
                                icon: 'success',
                                confirmButtonText: 'Aceptar'
                            }).then(() => {
                                $('#modal-emitir-nota-credito').modal('hide');
                                if (typeof table !== 'undefined' && table && table.ajax) {
                                    table.ajax.reload(null, false);
                                }
                            });
                        } else {
                            Swal.fire({
                                title: 'Error',
                                text: response.message || 'Error al emitir la Nota de Crédito',
                                icon: 'error',
                                confirmButtonText: 'Aceptar'
                            });
                        }
                    },
                    error: function(xhr, status, err) {
                        console.log('[NC] Error AJAX:', 'status', status, 'xhr.status', xhr && xhr.status, 'responseText', xhr && xhr.responseText ? xhr.responseText.substring(0, 300) : '', 'err', err);
                        addDebugLog('NC error: ' + (xhr ? xhr.status : '') + ' ' + (xhr && xhr.responseText ? xhr.responseText.substring(0, 200) : status), 'error');
                        Swal.close();
                        var errorMsg = 'Error al emitir la Nota de Crédito';
                        if (xhr.responseJSON && xhr.responseJSON.message) {
                            errorMsg = xhr.responseJSON.message;
                        } else if (xhr.responseText) {
                            try {
                                var j = JSON.parse(xhr.responseText);
                                if (j.message) errorMsg = j.message;
                            } catch (e) {}
                        }
                        Swal.fire({
                            title: 'Error',
                            text: errorMsg,
                            icon: 'error',
                            confirmButtonText: 'Aceptar'
                        });
                    }
                });
            }
        });
    });
});

function verDocumento(id) {
    $.ajax({
        url: $('#url').val() + 'facturacion/documentos_data',
        type: 'POST',
        data: { id_documento_fe: id },
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                var doc = response.data;
                var html = '<div class="row">';
                html += '<div class="col-md-6"><strong>ID:</strong> ' + doc.id_documento_fe + '</div>';
                html += '<div class="col-md-6"><strong>Fecha:</strong> ' + doc.fecha_creacion + '</div>';
                html += '<div class="col-md-6"><strong>Tipo:</strong> ' + doc.document_type + '</div>';
                html += '<div class="col-md-6"><strong>Serie:</strong> ' + doc.serie_electronica + '</div>';
                html += '<div class="col-md-6"><strong>Número:</strong> ' + String(doc.fd_number).padStart(10, '0') + '</div>';
                html += '<div class="col-md-6"><strong>POS:</strong> ' + doc.pos + '</div>';
                html += '<div class="col-md-12"><strong>Cliente:</strong> ' + doc.receptor_name + '</div>';
                html += '<div class="col-md-6"><strong>RUC:</strong> ' + (doc.receptor_ruc || 'N/A') + '</div>';
                html += '<div class="col-md-6"><strong>Email:</strong> ' + (doc.receptor_email || 'N/A') + '</div>';
                html += '<div class="col-md-6"><strong>Total:</strong> ' + parseFloat(doc.total).toFixed(2) + '</div>';
                html += '<div class="col-md-6"><strong>Estado:</strong> ' + doc.status + '</div>';
                if (doc.cufe) {
                    html += '<div class="col-md-12"><strong>CUFE:</strong> ' + doc.cufe + '</div>';
                }
                if (doc.authorization_number) {
                    html += '<div class="col-md-12"><strong>Número de Autorización:</strong> ' + doc.authorization_number + '</div>';
                }
                if ((doc.rejected == 1 || doc.status === 'rejected') && doc.rejected_messages) {
                    html += '<div class="col-md-12 mt-3"><strong class="text-danger">Motivo del rechazo (DGI):</strong><ul class="list-unstyled mt-2">';
                    try {
                        var msgs = typeof doc.rejected_messages === 'string' ? JSON.parse(doc.rejected_messages) : doc.rejected_messages;
                        if (Array.isArray(msgs)) {
                            msgs.forEach(function(m) {
                                html += '<li class="text-danger mb-1"><span class="badge badge-danger">' + (m.code || '') + '</span> ' + (m.message || '') + '</li>';
                            });
                        } else {
                            html += '<li class="text-danger">' + (msgs.message || doc.rejected_messages) + '</li>';
                        }
                    } catch (err) {
                        html += '<li class="text-muted">' + doc.rejected_messages + '</li>';
                    }
                    html += '</ul></div>';
                }
                html += '</div>';
                
                $('#contenido-documento').html(html);
                $('#modal-ver-documento').modal('show');
            } else {
                Swal.fire('Error', 'No se pudo cargar el documento', 'error');
            }
        },
        error: function() {
            Swal.fire('Error', 'Error al cargar el documento', 'error');
        }
    });
}

function enviarDocumento(id) {
    console.log('📤 enviarDocumento llamado con ID:', id);
    addDebugLog('Iniciando envío del documento ID: ' + id, 'request');
    
    if (!id || id === 'undefined' || id === '') {
        console.error('❌ ID inválido en enviarDocumento:', id);
        Swal.fire('Error', 'ID de documento inválido', 'error');
        return;
    }
    
    Swal.fire({
        title: 'Enviando documento...',
        text: 'Por favor espere',
        allowOutsideClick: false,
        didOpen: () => {
            Swal.showLoading();
        }
    });
    
    console.log('📤 Enviando petición AJAX para documento ID:', id);
    console.log('📤 URL:', $('#url').val() + 'facturacion/documentos_enviar');
    console.log('📤 Datos enviados:', { id_documento_fe: id });
    
    $.ajax({
        url: $('#url').val() + 'facturacion/documentos_enviar',
        type: 'POST',
        data: { id_documento_fe: id },
        dataType: 'json',
        beforeSend: function(xhr) {
            console.log('📤 Petición AJAX iniciada');
        },
        success: function(response) {
            console.log('📥 Respuesta completa de envío:', response);
            console.log('📥 Success:', response.success);
            console.log('📥 Message:', response.message);
            
            // Mostrar el payload si está disponible en la respuesta
            if (response.payload) {
                console.log('📥 Payload enviado (desde respuesta):', JSON.stringify(response.payload, null, 2));
            }
            if (response.header) {
                console.log('📥 Header enviado (desde respuesta):', JSON.stringify(response.header, null, 2));
            }
            
            Swal.close();
            if (response.success) {
                addDebugLog('Documento enviado exitosamente. ID: ' + id, 'success');
                Swal.fire({
                    title: '¡Enviado!',
                    text: response.message || 'Documento enviado correctamente',
                    icon: 'success',
                    confirmButtonText: 'Aceptar'
                }).then(() => {
                    // Cerrar modal si está abierto (solo para envío inicial, no para reenvío)
                    if ($('#modal-enviar-documento').is(':visible')) {
                        $('#modal-enviar-documento').modal('hide');
                    }
                    $('#table-documentos-fe').DataTable().ajax.reload();
                });
            } else {
                addDebugLog('Error al enviar documento: ' + (response.message || 'Error desconocido'), 'error');
                Swal.fire({
                    title: 'Error',
                    text: response.message || 'Error al enviar el documento',
                    icon: 'error',
                    confirmButtonText: 'Aceptar'
                });
            }
        },
        error: function(xhr, status, error) {
            console.error('❌ Error AJAX al enviar:', error, xhr);
            addDebugLog('Error de conexión al enviar documento: ' + error, 'error');
            Swal.close();
            Swal.fire('Error', 'Error de conexión al enviar el documento: ' + error, 'error');
        }
    });
}

function enviarPorEmail(id) {
    Swal.fire({
        title: 'Enviar por correo',
        input: 'email',
        inputLabel: 'Correo electrónico',
        inputPlaceholder: 'correo@ejemplo.com',
        showCancelButton: true,
        confirmButtonText: 'Enviar',
        cancelButtonText: 'Cancelar',
        inputValidator: (value) => {
            if (!value) {
                return 'Debe ingresar un correo electrónico';
            }
        }
    }).then((result) => {
        if (result.isConfirmed) {
            $.ajax({
                url: $('#url').val() + 'facturacion/documentos_enviar_email',
                type: 'POST',
                data: { 
                    id_documento_fe: id,
                    email: result.value
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        Swal.fire('¡Enviado!', 'El documento ha sido enviado por correo', 'success');
                    } else {
                        Swal.fire('Error', response.message || 'Error al enviar el correo', 'error');
                    }
                },
                error: function() {
                    Swal.fire('Error', 'Error de conexión', 'error');
                }
            });
        }
    });
}

